//----------------------------------------------------------------------
//
//  DBlock.cpp
//
//  Written by Matthew D Moss
//  1996, Ground Cover Software	
//
//  A class to encapsulate operations to a chunk of data, including
//  range replacement, insertion, and removal.  Multiple access will be
//  allowed (but shielded).
//
//----------------------------------------------------------------------


#include <string.h>
#include <Debug.h>
#include <Message.h>
#include "Acquire.h"
#include "DBlock.h"


//----------------------------------------------------------------------
DBlock::DBBuf::DBBuf ()
	: buf (NULL), used (0L), alloc (0L)
{
}


//----------------------------------------------------------------------
DBlock::DBBuf::DBBuf (const DBBuf& rhs)
	: buf (NULL), used (0L), alloc (0L)
{
	insert (0L, rhs.data (), rhs.size ());
}


//----------------------------------------------------------------------
DBlock::DBBuf::DBBuf (const char* data, long length)
	: buf (NULL), used (0L), alloc (0L)
{
	insert (0L, data, length);
}


//----------------------------------------------------------------------
DBlock::DBBuf::~DBBuf ()
{
	delete [] buf;
}


//----------------------------------------------------------------------
DBlock::DBBuf&
DBlock::DBBuf::operator = (const DBBuf& rhs)
{
	if (this != &rhs) {
		remove (0L, used);
		insert (0L, rhs.data (), rhs.size ());
	}
	
	return *this;
}


//----------------------------------------------------------------------
const char*
DBlock::DBBuf::data (void) const
{
	return buf;
}


//----------------------------------------------------------------------
long
DBlock::DBBuf::size (void) const
{
	return used;
}


//----------------------------------------------------------------------
void
DBlock::DBBuf::insert (long offset, const char* data, long length)
{
	if (length == 0L || data == NULL)
		return;

	ASSERT (length >  0L);
	ASSERT (offset >= 0L);
	ASSERT (offset <= used);

	long	newalloc = alloc;
	
	// determine if we need to allocate more memory
	while (newalloc < used + length) {
		newalloc += default_extend_alloc;
	}
	ASSERT (newalloc >= (used + length));

	if (newalloc != alloc) {
	
		// need to realloc then copy pieces into place
		char*	tmp;
		
		tmp = new char [newalloc];
		ASSERT (tmp != NULL);
		
		memcpy (tmp,					buf,			offset);
		memcpy (tmp + offset,			data,			length);
		memcpy (tmp + offset + length,	buf + offset,	used - offset);

		// get rid of the old and keep the new
		delete [] buf;
		buf = tmp;
		
		// update the allocated size;
		alloc = newalloc;
	}
	else {
	
		// enough already allocated, just copy pieces into place
		memmove (buf + offset + length,	buf + offset, 	used - offset);
		memcpy  (buf + offset,			data,			length);
	}
	
	// update the used space
	used += length;
}


//----------------------------------------------------------------------
void
DBlock::DBBuf::remove (long offset, long length)
{
	if (length == 0L)
		return;

	long	dx = offset + length;
	
	ASSERT (length >  0L);
	ASSERT (offset >= 0L);
	ASSERT (offset <  used);
	ASSERT (dx     <= used);

	memmove (buf + offset, buf + dx, used - dx);
	used -= length;
}


//----------------------------------------------------------------------
DBlock::DBlock ()
{
	mutex = create_sem(1L, "mutex");
}


//----------------------------------------------------------------------
DBlock::DBlock (const DBlock& rhs)
	: block (rhs.block)
{
	mutex = create_sem(1L, "mutex");
}


//----------------------------------------------------------------------
DBlock::DBlock (const char* data, long size)
	: block (data, size)
{
	mutex = create_sem(1L, "mutex");
}


//----------------------------------------------------------------------
DBlock::~DBlock ()
{
	(void) delete_sem(mutex);
}


//----------------------------------------------------------------------
DBlock&
DBlock::operator = (const DBlock& rhs)
{
	if (this != &rhs) {
		replace (0L, block.size (), rhs);
	}
	
	return *this;
}


//----------------------------------------------------------------------
DBlockRange
DBlock::operator () (long offset, long length)
{
	ASSERT (length >= 0L);
	ASSERT (offset >= 0L);
	ASSERT (offset + length <= block.size ());
	
	return DBlockRange (*this, offset, length);
}


//----------------------------------------------------------------------
const char*
DBlock::Data (long& size) const
{
	Acquire	lock(mutex);

	size = block.size ();
	return block.data ();
}


//----------------------------------------------------------------------
void
DBlock::replace (long offset, long length, const DBlock& rhs)
{
	Acquire	lock(mutex);

	const char*	rdata;
	long		rsize;
	
	rdata = rhs.Data (rsize);
	
	block.remove (offset, length);
	block.insert (offset, rdata, rsize);
	
	//	inform all those who have registered that the data has changed
	update_regd ();
}


//----------------------------------------------------------------------
DBlockRange::DBlockRange (const DBlockRange& rhs)
	: bas (rhs.bas), off (rhs.off), len (rhs.len)
{
}


//----------------------------------------------------------------------
DBlockRange::DBlockRange (DBlock& base, long offset, long length)
	: bas (base), off (offset), len (length)
{
}


//----------------------------------------------------------------------
DBlockRange&
DBlockRange::operator = (const DBlockRange& rhs)
{
	if (this != &rhs) {
		bas.replace (off, len, rhs);
	}
	
	return *this;
}


//----------------------------------------------------------------------
DBlockRange&
DBlockRange::operator = (const DBlock& rhs)
{
	if (this != &rhs) {
		bas.replace (off, len, rhs);
	}
	
	return *this;
}


//----------------------------------------------------------------------
const char*
DBlockRange::Data (long& size) const
{
	long	dummy;

	size = len;
	return (bas.Data (dummy) + off);
}


//----------------------------------------------------------------------
void
DBlockRange::replace (long offset, long length, const DBlock& rhs)
{
	bas.replace (off + offset, length, rhs);
}


//----------------------------------------------------------------------
bool
operator == (const DBlock::DBBuf& lhs, const DBlock::DBBuf& rhs)
{
	if (lhs.used != rhs.used)
		return FALSE;
	
	if (lhs.used == 0L)
		return TRUE;
	else
		return (memcmp (lhs.buf, rhs.buf, lhs.used) == 0);
}


//----------------------------------------------------------------------
bool
operator == (const DBlock& lhs, const DBlock& rhs)
{
	long		lsize;
	long		rsize;
	const char*	ldata;
	const char*	rdata;
	
	ldata = lhs.Data (lsize);
	rdata = rhs.Data (rsize);
	
	if (lsize != rsize)
		return FALSE;

	if (lsize == 0L)
		return TRUE;
	else
		return (memcmp (ldata, rdata, lsize) == 0);
}


//----------------------------------------------------------------------
void
DBlock::Register (BLooper* editor)
{
	ASSERT (editor != NULL);
	
	if (!regd.HasItem ( (void*)editor)) {
		regd.AddItem ( (void*)editor);
	}
}


//----------------------------------------------------------------------
void
DBlock::Unregister (BLooper* editor)
{
	ASSERT (editor != NULL);
	
	(void) regd.RemoveItem ( (void*)editor);
}


//----------------------------------------------------------------------
void
DBlock::update_regd (void)
{
	BMessage*	clone;
	BLooper*	editor;
	long		index = 0;
	
	clone = new BMessage (msg_DataChanged);
	
	while ( (editor = (BLooper*) regd.ItemAt (index)) != NULL) {
		editor->PostMessage (new BMessage (clone));
		++index;
	}
	
	delete clone;
}


//----------------------------------------------------------------------
void
DBlock::close_regd (void)
{
	BLooper*	editor;
	long		index = 0;
	
	while ( (editor = (BLooper*) regd.ItemAt (index)) != NULL) {
		editor->PostMessage (new BMessage (B_QUIT_REQUESTED));
		++index;
	}
}
